<?php

use App\Http\Controllers\ProfileController;
use App\Http\Controllers\StateController;
use App\Http\Controllers\CityController;
use App\Http\Controllers\DistrictController;
use App\Http\Controllers\DesignationController;
use App\Http\Controllers\MenuController;
use App\Http\Controllers\PermissionController;
use App\Http\Controllers\EmployeeController;
use App\Models\Permission;
use App\Helpers\RoleHelper;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use App\Models\User;
use App\Models\Role;


Route::get('/', function () {
    return view('welcome');
});

Route::get('/dashboard', function () {
    try {
        /** @var \App\Models\User $user */
        $user = Auth::user();

        if (!$user) {
            return redirect()->route('login');
        }

        $roleName = strtolower($user->getRoleName() ?? 'user');
        return redirect()->route($roleName . '.dashboard');
    } catch (\Exception $e) {
        Log::error('Dashboard route error: ' . $e->getMessage());
        return redirect()->route('login')->with('error', 'An error occurred. Please try again.');
    }
})->middleware(['auth', 'verified'])->name('dashboard');

// Dynamic role-based dashboard routes
$roles = Role::where('status', true)->get();
foreach ($roles as $role) {
    $roleName = strtolower($role->name);
    Route::middleware(['auth', 'verified'])->prefix($roleName)->name($roleName . '.')->group(function () use ($roleName) {
        Route::get('/dashboard', function () use ($roleName) {
            try {
                $user = Auth::user();
                if (!$user->hasRole($roleName)) {
                    abort(403, 'Unauthorized access.');
                }
                return view($roleName . '.dashboard');
            } catch (\Exception $e) {
                Log::error(ucfirst($roleName) . ' dashboard error: ' . $e->getMessage());
                return redirect()->route('dashboard')->with('error', 'Unable to load dashboard.');
            }
        })->name('dashboard');
    });
}

Route::middleware(['auth', 'permission'])->group(function () {

    // state Management Routes
    Route::get('/masters/location/state', [StateController::class, 'index'])->name('state.index');
    Route::post('/masters/location/state', [StateController::class, 'store'])->name('state.store');
    Route::put('/masters/location/state/{state}', [StateController::class, 'update'])->name('state.update');
    Route::delete('/masters/location/state/{state}', [StateController::class, 'destroy'])->name('state.destroy');
    Route::patch('/masters/location/state/{state}/toggle-status', [StateController::class, 'toggleStatus'])->name('state.toggle-status');

    // city Management Routes
    Route::get('/masters/location/city', [CityController::class, 'index'])->name('city.index');
    Route::post('/masters/location/city', [CityController::class, 'store'])->name('city.store');
    Route::put('/masters/location/city/{city}', [CityController::class, 'update'])->name('city.update');
    Route::delete('/masters/location/city/{city}', [CityController::class, 'destroy'])->name('city.destroy');
    Route::patch('/masters/location/city/{city}/toggle-status', [CityController::class, 'toggleStatus'])->name('city.toggle-status');
    Route::get('/api/districts/{state}', [EmployeeController::class, 'getDistricts'])->name('api.districts');

    // distict Management Routes
    Route::get('/masters/location/district', [DistrictController::class, 'index'])->name('district.index');
    Route::post('/masters/location/district', [DistrictController::class, 'store'])->name('district.store');
    Route::put('/masters/location/district/{district}', [DistrictController::class, 'update'])->name('district.update');
    Route::delete('/masters/location/district/{district}', [DistrictController::class, 'destroy'])->name('district.destroy');
    Route::patch('/masters/location/district/{district}/toggle-status', [DistrictController::class, 'toggleStatus'])->name('district.toggle-status');


    // Role Master Routes
    Route::get('/masters/user-role/designation', [DesignationController::class, 'index'])->name('designation.index');
    Route::post('/masters/user-role/designation', [DesignationController::class, 'store'])->name('designation.store');
    Route::put('/masters/user-role/designation/{designation}', [DesignationController::class, 'update'])->name('designation.update');
    Route::delete('/masters/user-role/designation/{designation}', [DesignationController::class, 'destroy'])->name('designation.destroy');
    Route::patch('/masters/user-role/designation/{designation}/toggle-status', [DesignationController::class, 'toggleStatus'])->name('designation.toggle-status');

    // Menu Management Routes
    Route::get('/masters/menu/menu', [MenuController::class, 'index'])->name('menu.index');
    Route::post('/masters/menu/menu', [MenuController::class, 'store'])->name('menu.store');
    Route::put('/masters/menu/menu/{menu}', [MenuController::class, 'update'])->name('menu.update');
    Route::delete('/masters/menu/menu/{menu}', [MenuController::class, 'destroy'])->name('menu.destroy');
    Route::patch('/masters/menu/menu/{menu}/toggle-status', [MenuController::class, 'toggleStatus'])->name('menu.toggle-status');

    // SubMenu Management Routes
    Route::get('/masters/menu/submenu', [MenuController::class, 'allSubMenus'])->name('submenu.all');
    Route::get('/masters/menu/{menu}/submenu', [MenuController::class, 'subMenus'])->name('submenu.index');
    Route::post('/masters/menu/{menu}/submenu', [MenuController::class, 'storeSubMenu'])->name('submenu.store');
    Route::put('/masters/menu/{menu}/submenu/{submenu}', [MenuController::class, 'updateSubMenu'])->name('submenu.update');
    Route::delete('/masters/menu/{menu}/submenu/{submenu}', [MenuController::class, 'destroySubMenu'])->name('submenu.destroy');
    Route::patch('/masters/menu/{menu}/submenu/{submenu}/toggle-status', [MenuController::class, 'toggleSubMenuStatus'])->name('submenu.toggle-status');

    // Permission Management Routes
    Route::get('/masters/user-role/permissions', [PermissionController::class, 'index'])->name('permissions.index');
    Route::post('/masters/user-role/permissions', [PermissionController::class, 'store'])->name('permissions.store');
    Route::get('/api/submenus/{menu}', [PermissionController::class, 'getSubMenus'])->name('api.submenus');
    Route::get('/api/role-permissions/{role}', [PermissionController::class, 'getRolePermissions'])->name('api.role-permissions');
    Route::delete('/api/role-permissions/{role}/reset', [PermissionController::class, 'resetRolePermissions'])->name('api.reset-permissions');


    // Team Management Routes
    Route::get('/team/add-employee', [EmployeeController::class, 'create'])->name('employee.create');
    Route::post('/team/add-employee', [EmployeeController::class, 'store'])->name('employee.store');
    Route::get('/team/employee-list', [EmployeeController::class, 'index'])->name('employee.index');
    Route::get('/team/employee/{employee}/edit', [EmployeeController::class, 'edit'])->name('employee.edit');
    Route::put('/team/employee/{employee}', [EmployeeController::class, 'update'])->name('employee.update');
    Route::delete('/team/employee/{employee}', [EmployeeController::class, 'destroy'])->name('employee.destroy');
    Route::patch('/team/employee/{employee}/toggle-status', [EmployeeController::class, 'toggleStatus'])->name('employee.toggle-status');
    Route::patch('/team/employee/{employee}/change-password', [EmployeeController::class, 'changePassword'])->name('employee.change-password');
    Route::get('/api/cities/{district}', [EmployeeController::class, 'getCities'])->name('api.cities');
    Route::get('/profile-settings', [EmployeeController::class, 'profileIndex'])->name('profile-settings.get');
    Route::post('/profile/update', [EmployeeController::class, 'profileUpdate'])->name('profile.update');
    Route::post('/profile/change-password', [EmployeeController::class, 'profileChangePassword'])->name('profile.change-password');
});

require __DIR__ . '/auth.php';
