document.addEventListener('DOMContentLoaded', function() {
    const roleSelect = document.getElementById('role_id');
    const menuCheckboxes = document.querySelectorAll('.menu-checkbox');
    const submenuContainer = document.getElementById('submenu-container');
    let allSubmenus = {}; // Store all submenus by menu ID
    let hasExistingPermissions = false; // Track if role has existing permissions

    roleSelect.addEventListener('change', function() {
        if (this.value) {
            loadRolePermissions(this.value);
        } else {
            clearPermissions();
        }
    });

    document.getElementById('resetPermissions').addEventListener('click', function() {
        const roleId = roleSelect.value;
        if (!roleId) {
            alert('Please select a role first');
            return;
        }
        
        if (confirm('Are you sure you want to reset all permissions for this role?')) {
            resetRolePermissions(roleId);
        }
    });

    document.getElementById('permissionsForm').addEventListener('submit', function(e) {
        if (!validateSubmenuPermissions()) {
            e.preventDefault();
            return false;
        }
    });

    menuCheckboxes.forEach(checkbox => {
        checkbox.addEventListener('change', function() {
            if (this.checked) {
                loadAndAppendSubMenus(this.dataset.menuId);
            } else {
                removeSubMenus(this.dataset.menuId);
            }
        });
    });

    function loadAndAppendSubMenus(menuId) {
        fetch(`/api/submenus/${menuId}`)
            .then(response => response.json())
            .then(submenus => {
                allSubmenus[menuId] = submenus;
                renderAllSubmenus(!hasExistingPermissions);
            });
    }

    function removeSubMenus(menuId) {
        delete allSubmenus[menuId];
        renderAllSubmenus();
    }

    function renderAllSubmenus(autoCheck = false) {
        let html = '';
        const checkedMenus = document.querySelectorAll('.menu-checkbox:checked');

        if (checkedMenus.length === 0) {
            submenuContainer.innerHTML = '<p class="text-muted">Select a menu to view its submenus</p>';
            return;
        }

        const currentlyChecked = Array.from(document.querySelectorAll('input[name="submenu_permissions[]"]:checked'))
            .map(cb => cb.value);

        checkedMenus.forEach(menuCheckbox => {
            const menuId = menuCheckbox.dataset.menuId;
            const menuName = menuCheckbox.nextElementSibling.textContent;

            if (allSubmenus[menuId]) {
                html += `<h6 class="text-primary mt-3 mb-2">${menuName}</h6>`;
                
                // Get parent submenus (no parent_id)
                const parentSubmenus = allSubmenus[menuId].filter(s => !s.parent_id);
                
                parentSubmenus.forEach(parentSubmenu => {
                    const wasChecked = currentlyChecked.includes(parentSubmenu.id.toString());
                    const shouldCheck = autoCheck || wasChecked ? 'checked' : '';
                    html += `
                        <div class="form-check mb-2 ms-3">
                            <input class="form-check-input" type="checkbox" 
                                   name="submenu_permissions[]" value="${parentSubmenu.id}" 
                                   id="submenu_${parentSubmenu.id}" ${shouldCheck}>
                            <label class="form-check-label fw-bold" for="submenu_${parentSubmenu.id}">
                                ${parentSubmenu.name}
                            </label>
                        </div>
                    `;
                    
                    // Get child submenus for this parent
                    const childSubmenus = allSubmenus[menuId].filter(s => s.parent_id === parentSubmenu.id);
                    childSubmenus.forEach(childSubmenu => {
                        const childWasChecked = currentlyChecked.includes(childSubmenu.id.toString());
                        const childShouldCheck = autoCheck || childWasChecked ? 'checked' : '';
                        html += `
                            <div class="form-check mb-2 ms-5">
                                <input class="form-check-input" type="checkbox" 
                                       name="submenu_permissions[]" value="${childSubmenu.id}" 
                                       id="submenu_${childSubmenu.id}" ${childShouldCheck}>
                                <label class="form-check-label" for="submenu_${childSubmenu.id}">
                                    └─ ${childSubmenu.name}
                                </label>
                            </div>
                        `;
                    });
                });
            }
        });

        submenuContainer.innerHTML = html || '<p class="text-muted">No submenus available</p>';
    }

    function renderAllSubmenus() {
        let html = '';
        const checkedMenus = document.querySelectorAll('.menu-checkbox:checked');

        if (checkedMenus.length === 0) {
            submenuContainer.innerHTML = '<p class="text-muted">Select a menu to view its submenus</p>';
            return;
        }

        const currentlyChecked = Array.from(document.querySelectorAll('input[name="submenu_permissions[]"]:checked'))
            .map(cb => cb.value);

        checkedMenus.forEach(menuCheckbox => {
            const menuId = menuCheckbox.dataset.menuId;
            const menuName = menuCheckbox.nextElementSibling.textContent;

            if (allSubmenus[menuId]) {
                html += `<h6 class="text-primary mt-3 mb-2">${menuName}</h6>`;
                
                // Get parent submenus (no parent_id)
                const parentSubmenus = allSubmenus[menuId].filter(s => !s.parent_id);
                
                parentSubmenus.forEach(parentSubmenu => {
                    const wasChecked = currentlyChecked.includes(parentSubmenu.id.toString());
                    html += `
                        <div class="form-check mb-2 ms-3">
                            <input class="form-check-input" type="checkbox" 
                                   name="submenu_permissions[]" value="${parentSubmenu.id}" 
                                   id="submenu_${parentSubmenu.id}" ${wasChecked ? 'checked' : ''}>
                            <label class="form-check-label fw-bold" for="submenu_${parentSubmenu.id}">
                                ${parentSubmenu.name}
                            </label>
                        </div>
                    `;
                    
                    // Get child submenus for this parent
                    const childSubmenus = allSubmenus[menuId].filter(s => s.parent_id === parentSubmenu.id);
                    childSubmenus.forEach(childSubmenu => {
                        const childWasChecked = currentlyChecked.includes(childSubmenu.id.toString());
                        html += `
                            <div class="form-check mb-2 ms-5">
                                <input class="form-check-input" type="checkbox" 
                                       name="submenu_permissions[]" value="${childSubmenu.id}" 
                                       id="submenu_${childSubmenu.id}" ${childWasChecked ? 'checked' : ''}>
                                <label class="form-check-label" for="submenu_${childSubmenu.id}">
                                    └─ ${childSubmenu.name}
                                </label>
                            </div>
                        `;
                    });
                });
            }
        });

        submenuContainer.innerHTML = html || '<p class="text-muted">No submenus available</p>';
    }

    function loadRolePermissions(roleId) {
        fetch(`/api/role-permissions/${roleId}`)
            .then(response => response.json())
            .then(data => {
                // Clear current selections
                clearPermissions();
                
                hasExistingPermissions = data.menu_ids.length > 0;
                
                // Check menu permissions
                data.menu_ids.forEach(menuId => {
                    const menuCheckbox = document.getElementById(`menu_${menuId}`);
                    if (menuCheckbox) {
                        menuCheckbox.checked = true;
                        loadAndAppendSubMenusWithExisting(menuId, data.submenu_ids);
                    }
                });
            });
    }

    function loadAndAppendSubMenusWithExisting(menuId, existingSubmenuIds) {
        fetch(`/api/submenus/${menuId}`)
            .then(response => response.json())
            .then(submenus => {
                allSubmenus[menuId] = submenus;
                renderAllSubmenusWithExisting(existingSubmenuIds);
            });
    }

    function renderAllSubmenusWithExisting(existingSubmenuIds = []) {
        let html = '';
        const checkedMenus = document.querySelectorAll('.menu-checkbox:checked');

        if (checkedMenus.length === 0) {
            submenuContainer.innerHTML = '<p class="text-muted">Select a menu to view its submenus</p>';
            return;
        }

        checkedMenus.forEach(menuCheckbox => {
            const menuId = menuCheckbox.dataset.menuId;
            const menuName = menuCheckbox.nextElementSibling.textContent;

            if (allSubmenus[menuId]) {
                html += `<h6 class="text-primary mt-3 mb-2">${menuName}</h6>`;
                
                // Get parent submenus (no parent_id)
                const parentSubmenus = allSubmenus[menuId].filter(s => !s.parent_id);
                
                parentSubmenus.forEach(parentSubmenu => {
                    const isChecked = existingSubmenuIds.includes(parentSubmenu.id.toString()) ? 'checked' : '';
                    html += `
                        <div class="form-check mb-2 ms-3">
                            <input class="form-check-input" type="checkbox" 
                                   name="submenu_permissions[]" value="${parentSubmenu.id}" 
                                   id="submenu_${parentSubmenu.id}" ${isChecked}>
                            <label class="form-check-label fw-bold" for="submenu_${parentSubmenu.id}">
                                ${parentSubmenu.name}
                            </label>
                        </div>
                    `;
                    
                    // Get child submenus for this parent
                    const childSubmenus = allSubmenus[menuId].filter(s => s.parent_id === parentSubmenu.id);
                    childSubmenus.forEach(childSubmenu => {
                        const childIsChecked = existingSubmenuIds.includes(childSubmenu.id.toString()) ? 'checked' : '';
                        html += `
                            <div class="form-check mb-2 ms-5">
                                <input class="form-check-input" type="checkbox" 
                                       name="submenu_permissions[]" value="${childSubmenu.id}" 
                                       id="submenu_${childSubmenu.id}" ${childIsChecked}>
                                <label class="form-check-label" for="submenu_${childSubmenu.id}">
                                    └─ ${childSubmenu.name}
                                </label>
                            </div>
                        `;
                    });
                });
            }
        });

        submenuContainer.innerHTML = html || '<p class="text-muted">No submenus available</p>';
    }

    function clearPermissions() {
        document.querySelectorAll('input[type="checkbox"]').forEach(cb => cb.checked = false);
        allSubmenus = {};
        hasExistingPermissions = false;
        submenuContainer.innerHTML = '<p class="text-muted">Select a menu to view its submenus</p>';
    }

    function resetRolePermissions(roleId) {
        fetch(`/api/role-permissions/${roleId}/reset`, {
            method: 'DELETE',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                'Content-Type': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                clearPermissions();
                alert('Permissions reset successfully');
            } else {
                alert('Error resetting permissions');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error resetting permissions');
        });
    }

    function validateSubmenuPermissions() {
        const checkedSubmenus = Array.from(document.querySelectorAll('input[name="submenu_permissions[]"]:checked'));
        
        for (let checkbox of checkedSubmenus) {
            const submenuId = checkbox.value;
            
            // Find the submenu data to check if it has a parent
            for (let menuId in allSubmenus) {
                const submenu = allSubmenus[menuId].find(s => s.id.toString() === submenuId);
                if (submenu && submenu.parent_id) {
                    // This is a child submenu, check if parent is selected
                    const parentCheckbox = document.getElementById(`submenu_${submenu.parent_id}`);
                    if (!parentCheckbox || !parentCheckbox.checked) {
                        alert(`Please select the parent submenu before selecting "${submenu.name}"`);
                        return false;
                    }
                }
            }
        }
        return true;
    }
});