class EmployeeValidation {
    constructor() {
        this.initValidation();
    }

    initValidation() {
        const emailField = document.getElementById('email');
        if (emailField) {
            let emailTimeout;
            emailField.addEventListener('input', (e) => {
                clearTimeout(emailTimeout);
                emailTimeout = setTimeout(() => {
                    this.validateEmail(e.target.value, e.target);
                }, 500);
            });
        }

        const mobileField = document.getElementById('mobile');
        if (mobileField) {
            let mobileTimeout;
            mobileField.addEventListener('input', (e) => {
                clearTimeout(mobileTimeout);
                mobileTimeout = setTimeout(() => {
                    this.validateMobile(e.target.value, e.target);
                }, 500);
            });
        }

        // const stateField = document.getElementById('state_id');
        // if (stateField) {
        //     stateField.addEventListener('change', (e) => {
        //         this.loadCities(e.target.value);
        //     });
        // }

        const stateField = document.getElementById('state_id');
        if (stateField) {
            stateField.addEventListener('change', (e) => {
                this.loadDistricts(e.target.value);
            });
        }

        const districtField = document.getElementById('district_id');
        if (districtField) {
            districtField.addEventListener('change', (e) => {
                this.loadCities(e.target.value);
            });
        }

        const togglePassword = document.getElementById('togglePassword');
        if (togglePassword) {
            togglePassword.addEventListener('click', this.togglePasswordVisibility);
        }
    }

    getCSRFToken() {
        const csrfInput = document.querySelector('input[name="_token"]');
        if (csrfInput && csrfInput.value) {
            return csrfInput.value;
        }

        const csrfMeta = document.querySelector('meta[name="csrf-token"]');
        if (csrfMeta) {
            return csrfMeta.content || csrfMeta.getAttribute('content');
        }

        return null;
    }

    async validateEmail(email, field) {
        if (!email || email.length < 3) {
            this.clearValidation(field);
            return;
        }

        const token = this.getCSRFToken();
        if (!token) {
            console.error('CSRF token not found');
            return;
        }

        try {
            const isProfile = window.location.pathname.includes('profile');
            const endpoint = isProfile ? '/api/profile/check-email' : '/api/check-email';

            const response = await fetch(endpoint, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': token
                },
                body: JSON.stringify({
                    email: email,
                    user_id: document.getElementById('user_id') ? document.getElementById('user_id').value : null
                })
            });

            const data = await response.json();
            this.showValidationResult(field, data.exists, 'Email already exists');
        } catch (error) {
            console.error('Email validation error:', error);
        }
    }

    async validateMobile(mobile, field) {
        if (!mobile || mobile.length < 10) {
            this.clearValidation(field);
            return;
        }

        const token = this.getCSRFToken();
        if (!token) {
            console.error('CSRF token not found');
            return;
        }

        try {
            const isProfile = window.location.pathname.includes('profile');
            const endpoint = isProfile ? '/api/profile/check-mobile' : '/api/check-mobile';

            const response = await fetch(endpoint, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': token
                },
                body: JSON.stringify({
                    mobile: mobile,
                    user_id: document.getElementById('user_id') ? document.getElementById('user_id').value : null
                })
            });

            const data = await response.json();
            this.showValidationResult(field, data.exists, 'Mobile number already exists');
        } catch (error) {
            console.error('Mobile validation error:', error);
        }
    }

    showValidationResult(field, exists, message) {
        const feedback = field.parentElement.querySelector('.validation-feedback') ||
            this.createFeedbackElement(field);

        if (exists) {
            field.classList.add('is-invalid');
            field.classList.remove('is-valid');
            feedback.textContent = message;
            feedback.className = 'validation-feedback text-danger';
            feedback.style.display = 'block';
        } else {
            field.classList.add('is-valid');
            field.classList.remove('is-invalid');
            feedback.textContent = 'Available';
            feedback.className = 'validation-feedback text-success';
            feedback.style.display = 'block';
        }
    }

    clearValidation(field) {
        field.classList.remove('is-valid', 'is-invalid');
        const feedback = field.parentElement.querySelector('.validation-feedback');
        if (feedback) {
            feedback.style.display = 'none';
        }
    }

    createFeedbackElement(field) {
        const feedback = document.createElement('div');
        feedback.className = 'validation-feedback';
        field.parentElement.appendChild(feedback);
        return feedback;
    }

    // async loadCities(stateId) {
    //     alert(stateId);
    //     const citySelect = document.getElementById('city_id');
    //     if (!citySelect) return;

    //     citySelect.innerHTML = '<option value="">Select City</option>';

    //     if (stateId) {
    //         try {
    //             const response = await fetch(`/api/cities/${stateId}`);
    //             const cities = await response.json();

    //             cities.forEach(city => {
    //                 citySelect.innerHTML += `<option value="${city.id}">${city.name}</option>`;
    //             });
    //         } catch (error) {

    //             console.error('Error loading cities:', error);
    //         }
    //     }
    // }


    async loadDistricts(stateId) {
        const districtSelect = document.getElementById('district_id');
        const citySelect = document.getElementById('city_id');
        
        if (!districtSelect) return;

        districtSelect.innerHTML = '<option value="">Select District</option>';
        if (citySelect) {
            citySelect.innerHTML = '<option value="">Select City</option>';
        }

        if (!stateId) return;

        try {
            const response = await fetch(`/api/districts/${stateId}`);
            const result = await response.json();

            if (result.success) {
                result.data.forEach(district => {
                    districtSelect.innerHTML +=
                        `<option value="${district.id}">${district.name}</option>`;
                });
            }
        } catch (error) {
            console.error('Error loading districts:', error);
        }
    }

    async loadCities(districtId) {
        const citySelect = document.getElementById('city_id');
        if (!citySelect) return;

        citySelect.innerHTML = '<option value="">Select City</option>';

        if (!districtId) return;

        try {
            const response = await fetch(`/api/cities/${districtId}`);
            const result = await response.json();

            if (result.success) {
                result.data.forEach(city => {
                    citySelect.innerHTML +=
                        `<option value="${city.id}">${city.name}</option>`;
                });
            }
        } catch (error) {
            console.error('Error loading cities:', error);
        }
    }


    togglePasswordVisibility() {
        const passwordField = document.getElementById('password');
        const eyeIcon = document.getElementById('eyeIcon');

        if (passwordField && eyeIcon) {
            if (passwordField.type === 'password') {
                passwordField.type = 'text';
                eyeIcon.className = 'ri-eye-off-line';
            } else {
                passwordField.type = 'password';
                eyeIcon.className = 'ri-eye-line';
            }
        }
    }
}

document.addEventListener('DOMContentLoaded', function () {
    new EmployeeValidation();
});