<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\Log;
use Laravel\Sanctum\HasApiTokens;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable, SoftDeletes, HasApiTokens;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'father_name',
        'mother_name',
        'dob',
        'gender',
        'marital_status',
        'email',
        'password',
        'mobile',
        'alternate_mobile',
        'aadhaar_number',
        'pan_number',
        'staff_id',
        'role_id',
        'department_id',
        'branch_id',
        'joining_date',
        'status',
        'profile_pic',
        'addhar_front_pic',
        'addhar_back_pic',
        'state_id',
        'district_id',
        'city_id',
        'address',
        'pincode',
        'is_location_enable',
        'device_info',
        'device_id',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            // amazonq-ignore-next-line
            'password' => 'hashed',
            'dob' => 'date',
            'joining_date' => 'date',
            'is_location_enable' => 'boolean',
            'device_info' => 'json',
        ];
    }

    /**
     * Check if user has specific role by name
     */
    public function hasRole(string $roleName): bool
    {
        return $this->role && strtolower($this->role->name) === strtolower($roleName);
    }

    /**
     * Check if user has specific role by ID
     */
    public function hasRoleId($roleId): bool
    {
        return $this->role_id == $roleId;
    }

    /**
     * Get user role name
     */
    public function getRoleName(): ?string
    {
        return $this->role ? $this->role->name : null;
    }

    /**
     * Check if user is admin (backward compatibility)
     */
    public function isAdmin(): bool
    {
        return $this->hasRole('admin');
    }

    /**
     * Check if user is regular user (backward compatibility)
     */
    public function isUser(): bool
    {
        return $this->hasRole('user');
    }

    /**
     * Get all attendance records for this user
     * Relationship: One User has Many Attendances (based on staff_id)
     */
    public function attendances()
    {
        return $this->hasMany(Attendance::class, 'staff_id', 'staff_id');
    }

    /**
     * Get all regularize records for this user
     * Relationship: One User has Many Regularizes (based on staff_id)
     */
    public function regularizes()
    {
        return $this->hasMany(Regularize::class, 'staff_id', 'staff_id');
    }

    /**
     * Get the department that owns the user
     */
    public function department()
    {
        return $this->belongsTo(Department::class, 'department_id');
    }

    /**
     * Get the branch that owns the user
     */
    public function branch()
    {
        return $this->belongsTo(Branch::class, 'branch_id');
    }

    /**
     * Get the role that owns the user
     */
    public function role()
    {
        return $this->belongsTo(Role::class, 'role_id');
    }

    /**
     * Get the state that owns the user
     */
    public function state()
    {
        return $this->belongsTo(State::class, 'state_id');
    }

    /**
     * Get the city that owns the user
     */
    public function city()
    {
        return $this->belongsTo(City::class, 'city_id');
    }

    /**
     * Get the district that owns the user
     */
    public function district()
    {
        return $this->belongsTo(District::class, 'district_id');
    }

    /**
     * Get user permissions with menus and submenus
     */
    public function getPermissions()
    {
        // amazonq-ignore-next-line
        return Permission::with('menu')
            ->where('role_id', $this->role_id)
            ->get();
    }

    /**
     * Check if user has permission for a specific menu
     */
    public function hasMenuPermission($menuId)
    {
        return Permission::where('role_id', $this->role_id)
            ->where('menu_id', $menuId)
            ->exists();
    }

    /**
     * Check if user has permission for a specific submenu
     */
    public function hasSubmenuPermission($submenuId)
    {
        $permissions = Permission::where('role_id', $this->role_id)->get();
        
        foreach ($permissions as $permission) {
            if ($permission->sub_menu) {
                $submenuIds = explode(',', $permission->sub_menu);
                if (in_array($submenuId, $submenuIds)) {
                    return true;
                }
            }
        }
        
        return false;
    }

    /**
     * Check if user has permission for a specific route
     */
    public function hasRoutePermission($routeName)
    {
        try {
            // First try to find by route name
            $submenu = SubMenu::where('route', $routeName)->first();
            
            // If not found by route name, try to find by route path
            if (!$submenu) {
                $currentPath = request()->getPathInfo();
                $submenu = SubMenu::where('route', $currentPath)->first();
            }
            
            if (!$submenu) {
                return true; // Allow access if route is not in submenu system
            }
            
            return $this->hasSubmenuPermission($submenu->id);
        } catch (\Exception $e) {
            Log::error('Route permission check error: ' . $e->getMessage());
            return false; // Deny access on error
        }
    }
}
