<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use App\Models\FormField;
use App\Models\FormFieldOption;
use App\Models\Product;

class ProductController extends Controller
{
    public function index()
    {
        try {
            $products = Product::latest()->paginate(10);
            $coDesignOptions = FormFieldOption::whereHas('formField', function($query) {
                $query->where('field_name', 'co_deisgn');
            })->pluck('option_value', 'id');
            return view('admin.product.index', compact('products', 'coDesignOptions'));
        } catch (\Exception $e) {
            Log::error('Error loading products: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Error loading products.');
        }
    }

    public function create()
    {
        try {
            $dropdownFields = FormField::with('options')->get()->keyBy('field_name');
            return view('admin.product.create', compact('dropdownFields'));
        } catch (\Exception $e) {
            Log::error('Error in FormFieldController@create: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while loading the create form.');
        }
    }

    public function store(Request $request)
    {
        try {
            // Handle file upload
            $cadPhotoPath = null;
            if ($request->hasFile('cad_photo')) {
                $cadPhotoPath = $request->file('cad_photo')->store('cad_photos', 'public');
            }

            // Prepare yarn shade data and calculate totals
            $yarnShadeData = [];
            $totalPercentage = 0;
            $totalKilogram = 0;
            
            for ($i = 1; $i <= 20; $i++) {
                if ($request->input("yarn.{$i}") && $request->input("shade.{$i}")) {
                    $percentage = (float) ($request->input("percentage.{$i}") ?? 0);
                    $kilogram = (float) ($request->input("kilogram.{$i}") ?? 0);
                    
                    $yarnShadeData[] = [
                        'yarn_id' => $request->input("yarn.{$i}"),
                        'shade_id' => $request->input("shade.{$i}"),
                        'percentage' => $percentage,
                        'kilogram' => $kilogram,
                    ];
                    
                    $totalPercentage += $percentage;
                    $totalKilogram += $kilogram;
                }
            }

            // Create product
            Product::create([
                'product_name' => $request->product_name,
                'product_card_no' => $request->product_card_no,
                'co_design' => $request->co_design,
                'standard_size' => $request->standard_size,
                'color' => $request->color,
                'collection' => $request->collection,
                'shape' => $request->shape,
                'style' => $request->style,
                'customer_design' => $request->customer_design,
                'quality' => $request->quality,
                'weave' => $request->weave,
                'cad_photo' => $cadPhotoPath,
                'yarn_shade_data' => $yarnShadeData,
                'total_percentage' => $totalPercentage,
                'total_kilogram' => $totalKilogram
            ]);

            return redirect()->route('product.create')->with('success', 'Product card saved successfully!');
        } catch (\Exception $e) {
            Log::error('Error saving product: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Error saving product card.');
        }
    }

    public function show($id)
    {
        try {
            $product = Product::findOrFail($id);
            $dropdownFields = FormField::with('options')->get()->keyBy('field_name');
            return view('admin.product.show', compact('product', 'dropdownFields'));
        } catch (\Exception $e) {
            Log::error('Error loading product details: ' . $e->getMessage());
            return redirect()->route('product.index')->with('error', 'Product not found.');
        }
    }

    public function edit($id)
    {
        try {
            $product = Product::findOrFail($id);
            $dropdownFields = FormField::with('options')->get()->keyBy('field_name');
            return view('admin.product.edit', compact('product', 'dropdownFields'));
        } catch (\Exception $e) {
            Log::error('Error loading product for edit: ' . $e->getMessage());
            return redirect()->route('product.index')->with('error', 'Product not found.');
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $product = Product::findOrFail($id);
            
            // Handle file upload
            $cadPhotoPath = $product->cad_photo;
            if ($request->hasFile('cad_photo')) {
                $cadPhotoPath = $request->file('cad_photo')->store('cad_photos', 'public');
            }

            // Prepare yarn shade data and calculate totals
            $yarnShadeData = [];
            $totalPercentage = 0;
            $totalKilogram = 0;
            
            for ($i = 1; $i <= 20; $i++) {
                if ($request->input("yarn.{$i}") && $request->input("shade.{$i}")) {
                    $percentage = (float) ($request->input("percentage.{$i}") ?? 0);
                    $kilogram = (float) ($request->input("kilogram.{$i}") ?? 0);
                    
                    $yarnShadeData[] = [
                        'yarn_id' => $request->input("yarn.{$i}"),
                        'shade_id' => $request->input("shade.{$i}"),
                        'percentage' => $percentage,
                        'kilogram' => $kilogram,
                    ];
                    
                    $totalPercentage += $percentage;
                    $totalKilogram += $kilogram;
                }
            }

            // Update product
            $product->update([
                'product_name' => $request->product_name,
                'co_design' => $request->co_design,
                'standard_size' => $request->standard_size,
                'color' => $request->color,
                'collection' => $request->collection,
                'shape' => $request->shape,
                'style' => $request->style,
                'customer_design' => $request->customer_design,
                'quality' => $request->quality,
                'weave' => $request->weave,
                'cad_photo' => $cadPhotoPath,
                'yarn_shade_data' => $yarnShadeData,
                'total_percentage' => $totalPercentage,
                'total_kilogram' => $totalKilogram
            ]);

            return redirect()->route('product.index')->with('success', 'Product updated successfully!');
        } catch (\Exception $e) {
            Log::error('Error updating product: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Error updating product.');
        }
    }

    public function destroy($id)
    {
        //
    }

    public function getShadeOptions($yarnId)
    {
        try {
            $shadeOptions = FormFieldOption::where('parent_id', $yarnId)
                ->where('status', true)
                ->get(['id', 'option_value']);
            
            return response()->json($shadeOptions);
        } catch (\Exception $e) {
            Log::error('Error fetching shade options: ' . $e->getMessage());
            return response()->json([], 500);
        }
    }
}