<?php

namespace App\Http\Controllers;

use App\Models\Permission;
use App\Models\Role;
use App\Models\Menu;
use App\Models\SubMenu;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class PermissionController extends Controller
{
    public function index()
    {
        try {
            $roles = Role::whereNull('deleted_at')->where('status', 1)->get();
            $menus = Menu::whereNull('deleted_at')->where('status', 1)->orderBy('priority')->get();
            $permissions = Permission::whereNull('deleted_at')->get()->keyBy('role_id');

            // Get role permissions with hierarchical structure
            $rolePermissions = [];
            foreach ($roles as $role) {
                $rolePermissions[$role->id] = Permission::where('role_id', $role->id)
                    ->with('menu')
                    ->get()
                    ->map(function ($permission) {
                        $submenuData = null;
                        if ($permission->sub_menu) {
                            $submenuIds = explode(',', $permission->sub_menu);
                            $submenus = SubMenu::whereIn('id', $submenuIds)->orderBy('priority')->get();
                            $submenuData = [
                                'parent' => $submenus->whereNull('parent_id'),
                                'child' => $submenus->whereNotNull('parent_id')->groupBy('parent_id')
                            ];
                        }
                        return (object) [
                            'menu' => $permission->menu,
                            'submenu_data' => $submenuData
                        ];
                    });
            }

            return view('masters.user-role.permissions', compact('roles', 'menus', 'permissions', 'rolePermissions'));
        } catch (\Exception $e) {
            Log::error('Error in PermissionController@index: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while fetching permissions.');
        }
    }

    public function store(Request $request)
    {
        try {
            $request->validate([
                'role_id' => 'required|exists:roles,id',
                'menu_permissions' => 'array',
                'submenu_permissions' => 'array'
            ]);

            DB::transaction(function () use ($request) {
                $menuPermissions = $request->menu_permissions ?? [];
                $submenuPermissions = $request->submenu_permissions ?? [];

                // Group submenus by their menu_id
                $submenusByMenu = [];
                if (!empty($submenuPermissions)) {
                    $submenus = SubMenu::whereIn('id', $submenuPermissions)->get();
                    foreach ($submenus as $submenu) {
                        $submenusByMenu[$submenu->menu_id][] = $submenu->id;
                    }
                }

                // Get existing permissions for this role
                $existingPermissions = Permission::where('role_id', $request->role_id)->get()->keyBy('menu_id');

                // Update or create permissions for each menu
                foreach ($menuPermissions as $menuId) {
                    $submenuIds = $submenusByMenu[$menuId] ?? [];

                    if (isset($existingPermissions[$menuId])) {
                        // Update existing permission
                        $existingPermissions[$menuId]->update([
                            'sub_menu' => implode(',', $submenuIds),
                            'sub_child_menu' => ''
                        ]);
                    } else {
                        // Create new permission
                        Permission::create([
                            'role_id' => $request->role_id,
                            'menu_id' => $menuId,
                            'sub_menu' => implode(',', $submenuIds),
                            'sub_child_menu' => ''
                        ]);
                    }
                }

                // Delete permissions for menus that are no longer selected
                $menusToDelete = $existingPermissions->keys()->diff($menuPermissions);
                Permission::where('role_id', $request->role_id)
                    ->whereIn('menu_id', $menusToDelete)
                    ->delete();
            });

            return redirect()->back()->with('success', 'Permissions updated successfully');
        } catch (\Exception $e) {
            Log::error('Error in PermissionController@store: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while updating permissions.');
        }
    }

    public function getSubMenus($menuId)
    {
        try {
            $subMenus = SubMenu::where('menu_id', $menuId)
                ->whereNull('deleted_at')
                ->where('status', 1)
                ->orderBy('priority')
                ->get();

            return response()->json($subMenus);
        } catch (\Exception $e) {
            Log::error('Error in PermissionController@getSubMenus: ' . $e->getMessage());
            return response()->json(['error' => 'An error occurred while fetching submenus.'], 500);
        }
    }

    public function getRolePermissions($roleId)
    {
        try {
            $permissions = Permission::where('role_id', $roleId)
                ->whereNull('deleted_at')
                ->get();

            $menuIds = $permissions->pluck('menu_id')->toArray();
            $submenuIds = [];

            foreach ($permissions as $permission) {
                if ($permission->sub_menu) {
                    $submenuIds = array_merge($submenuIds, explode(',', $permission->sub_menu));
                }
            }

            return response()->json([
                'menu_ids' => $menuIds,
                'submenu_ids' => array_filter($submenuIds)
            ]);
        } catch (\Exception $e) {
            Log::error('Error in PermissionController@getRolePermissions: ' . $e->getMessage());
            return response()->json(['error' => 'An error occurred while fetching role permissions.'], 500);
        }
    }

    public function resetRolePermissions($roleId)
    {
        try {
            DB::transaction(function () use ($roleId) {
                Permission::where('role_id', $roleId)->delete();
            });

            return response()->json(['success' => true, 'message' => 'Permissions reset successfully']);
        } catch (\Exception $e) {
            Log::error('Error in PermissionController@resetRolePermissions: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => 'An error occurred while resetting permissions.'], 500);
        }
    }
}
