<?php

namespace App\Http\Controllers;

use App\Models\FormField;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class FormFieldController extends Controller
{
    public function index()
    {
        try {
            $formFields = FormField::with('options')->orderBy('field_label')->orderBy('field_name')->get();
            return view('admin.master.form-fields.index', compact('formFields'));
        } catch (\Exception $e) {
            Log::error('Error in FormFieldController@index: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while fetching form fields.');
        }
    }

    public function create()
    {
        try {
            return view('admin.master.form-fields.create');
        } catch (\Exception $e) {
            Log::error('Error in FormFieldController@create: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while loading the create form.');
        }
    }

    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'field_name' => 'required|string|max:100|unique:form_fields,field_name',
                'field_label' => 'required|string|max:150',
                'field_type' => 'required|in:select,radio,checkbox',
                'master_type' => 'nullable|in:master,sub_master',
                'parent_field_id' => 'nullable|exists:form_fields,id',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator)->withInput();
            }

            $data = $request->only(['field_label', 'field_type', 'master_type', 'parent_field_id', 'status']);
            $data['field_name'] = strtolower(str_replace(' ', '_', $request->field_name));

            FormField::create($data);
            return redirect()->route('form-fields.index')->with('success', 'Form field created successfully');
        } catch (\Exception $e) {
            Log::error('Error in FormFieldController@store: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while creating the form field.');
        }
    }

    public function edit(FormField $formField)
    {
        try {
            return view('admin.master.form-fields.edit', compact('formField'));
        } catch (\Exception $e) {
            Log::error('Error in FormFieldController@edit: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while loading the edit form.');
        }
    }

    public function update(Request $request, FormField $formField)
    {
        try {
            $validator = Validator::make($request->all(), [
                'field_name' => 'required|string|max:100|unique:form_fields,field_name,' . $formField->id,
                'field_label' => 'required|string|max:150',
                'field_type' => 'required|in:select,radio,checkbox',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator)->withInput();
            }

            $data = $request->only(['field_label', 'field_type', 'status']);
            $data['field_name'] = strtolower(str_replace(' ', '_', $request->field_name));

            $formField->update($data);
            return redirect()->route('form-fields.index')->with('success', 'Form field updated successfully');
        } catch (\Exception $e) {
            Log::error('Error in FormFieldController@update: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while updating the form field.');
        }
    }

    public function destroy(FormField $formField)
    {
        try {
            DB::transaction(function () use ($formField) {
                $formField->delete();
            });
            return redirect()->route('form-fields.index')->with('success', 'Form field deleted successfully');
        } catch (\Exception $e) {
            Log::error('Error in FormFieldController@destroy: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while deleting the form field.');
        }
    }
}
