<?php

namespace App\Http\Controllers;

use App\Models\District;
use App\Models\State;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class DistrictController extends Controller
{
    public function index()
    {
        try {
            $districts = District::with('state')->whereNull('deleted_at')->paginate(10);
            $states = State::whereNull('deleted_at')->where('status', 1)->get();
            return view('masters.location.district', compact('districts', 'states'));
        } catch (\Exception $e) {
            Log::error('Error in DistrictController@index: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while fetching districts.');
        }
    }

    public function store(Request $request)
    {
        try {
            $request->validate([
                'district_name' => 'required|string|max:255|unique:districts,name,NULL,id,state_id,' . $request->state_id . ',deleted_at,NULL',
                'state_id' => 'required|exists:states,id'
            ]);

            District::create([
                'name' => $request->district_name,
                'state_id' => $request->state_id
            ]);
            
            return redirect()->back()->with('success', 'District created successfully');
        } catch (\Exception $e) {
            Log::error('Error in DistrictController@store: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while creating the district.');
        }
    }

    public function update(Request $request, District $district)
    {
        try {
            $request->validate([
                'district_name' => 'required|string|max:255|unique:districts,name,' . $district->id . ',id,state_id,' . $request->state_id . ',deleted_at,NULL',
                'state_id' => 'required|exists:states,id'
            ]);

            $district->update([
                'name' => $request->district_name,
                'state_id' => $request->state_id
            ]);

            return redirect()->back()->with('success', 'District updated successfully');
        } catch (\Exception $e) {
            Log::error('Error in DistrictController@update: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while updating the district.');
        }
    }

    public function destroy(District $district)
    {
        try {
            $district->delete();
            return redirect()->back()->with('success', 'District deleted successfully');
        } catch (\Exception $e) {
            Log::error('Error in DistrictController@destroy: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while deleting the district.');
        }
    }

    public function toggleStatus(District $district)
    {
        try {
            $district->update(['status' => !$district->status]);
            $status = $district->status ? 'activated' : 'deactivated';
            return redirect()->back()->with('success', "District {$status} successfully");
        } catch (\Exception $e) {
            Log::error('Error in DistrictController@toggleStatus: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while updating district status.');
        }
    }
}