<?php

namespace App\Http\Controllers;

use App\Models\City;
use App\Models\State;
use App\Models\District;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class CityController extends Controller
{
    public function index()
    {
        try {
            $cities = City::with(['state', 'district'])->whereNull('deleted_at')->paginate(10);
            $states = State::whereNull('deleted_at')->where('status', 1)->get();
            return view('masters.location.city', compact('cities', 'states'));
        } catch (\Exception $e) {
            Log::error('Error in CityController@index: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while fetching cities.');
        }
    }

    public function getDistricts($stateId)
    {
        try {
            $districts = District::where('state_id', $stateId)->whereNull('deleted_at')->where('status', 1)->get();
            return response()->json($districts);
        } catch (\Exception $e) {
            Log::error('Error in CityController@getDistricts: ' . $e->getMessage());
            return response()->json(['error' => 'An error occurred while fetching districts.'], 500);
        }
    }



    public function store(Request $request)
    {
        try {
            $request->validate([
                'city_name' => [
                    'required',
                    'string',
                    'max:255',
                    function ($attribute, $value, $fail) use ($request) {
                        $exists = City::where('name', $value)
                            ->where('district_id', $request->district_id)
                            ->whereNull('deleted_at')
                            ->exists();
                        if ($exists) {
                            $fail('This city name already exists in the selected district.');
                        }
                    }
                ],
                'state_id' => 'required|exists:states,id',
                'district_id' => 'required|exists:districts,id'
            ]);

            City::create([
                'name' => $request->city_name,
                'state_id' => $request->state_id,
                'district_id' => $request->district_id
            ]);
            
            return redirect()->back()->with('success', 'City created successfully');
        } catch (\Exception $e) {
            Log::error('Error in CityController@store: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while creating the city.');
        }
    }

    public function update(Request $request, City $city)
    {
        try {
            $request->validate([
                'city_name' => [
                    'required',
                    'string',
                    'max:255',
                    function ($attribute, $value, $fail) use ($request, $city) {
                        $exists = City::where('name', $value)
                            ->where('district_id', $request->district_id)
                            ->where('id', '!=', $city->id)
                            ->whereNull('deleted_at')
                            ->exists();
                        if ($exists) {
                            $fail('This city name already exists in the selected district.');
                        }
                    }
                ],
                'state_id' => 'required|exists:states,id',
                'district_id' => 'required|exists:districts,id'
            ]);

            $city->update([
                'name' => $request->city_name,
                'state_id' => $request->state_id,
                'district_id' => $request->district_id
            ]);

            return redirect()->back()->with('success', 'City updated successfully');
        } catch (\Exception $e) {
            Log::error('Error in CityController@update: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while updating the city.');
        }
    }

    public function destroy(City $city)
    {
        try {
            $city->delete();
            return redirect()->back()->with('success', 'City deleted successfully');
        } catch (\Exception $e) {
            Log::error('Error in CityController@destroy: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while deleting the city.');
        }
    }

    public function toggleStatus(City $city)
    {
        try {
            $city->update(['status' => !$city->status]);
            $status = $city->status ? 'activated' : 'deactivated';
            return redirect()->back()->with('success', "City {$status} successfully");
        } catch (\Exception $e) {
            Log::error('Error in CityController@toggleStatus: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while updating city status.');
        }
    }
}