<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Branch;
use App\Models\State;
use App\Models\Role;
use App\Models\Department;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class BranchEmployeeController extends Controller
{
    public function index()
    {
        try {
            $user = Auth::user();
            
            // If admin (role_id = 1), show all employees, otherwise filter by branch
            if ($user->role_id == 1) {
                $employees = User::with(['role', 'department', 'branch'])
                    ->whereNull('deleted_at')
                    ->paginate(10);
            } else {
                $employees = User::with(['role', 'department', 'branch'])
                    ->where('branch_id', $user->branch_id)
                    ->whereNull('deleted_at')
                    ->paginate(10);
            }
                
            return view('masters.branch.branch-employees', compact('employees'));
        } catch (\Exception $e) {
            Log::error('Error in BranchEmployeeController@index: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while fetching employees.');
        }
    }

    public function create()
    {
        try {
            $user = Auth::user();
            $userRole = Role::find($user->role_id);
            
            // Filter roles based on user's authority level
            if ($userRole && $userRole->authorities) {
                // User can only assign roles with higher authority numbers (lower privilege)
                $roles = Role::whereNull('deleted_at')
                    ->where('authorities', '>', $userRole->authorities)
                    ->get();
            } else {
                // If user has no authority level set, show all roles
                $roles = Role::whereNull('deleted_at')->get();
            }
            
            $departments = Department::whereNull('deleted_at')->where('status', 1)->get();
            $branch = Branch::find($user->branch_id);
            $states = State::whereNull('deleted_at')->get();
            
            return view('masters.branch.add-branch-employee', compact('roles', 'departments', 'branch', 'states'));
        } catch (\Exception $e) {
            Log::error('Error in BranchEmployeeController@create: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while loading the create form.');
        }
    }

    public function edit(User $employee)
    {
        try {
            $user = Auth::user();
            
            // Check if employee belongs to same branch (skip check for admin)
            if ($user->role_id != 1 && $employee->branch_id !== $user->branch_id) {
                return redirect()->back()->with('error', 'Unauthorized access');
            }
            
            $userRole = Role::find($user->role_id);
            
            // Filter roles based on user's authority level
            if ($userRole && $userRole->authorities) {
                // User can only assign roles with higher authority numbers (lower privilege)
                $roles = Role::whereNull('deleted_at')
                    ->where('authorities', '>', $userRole->authorities)
                    ->get();
            } else {
                // If user has no authority level set, show all roles
                $roles = Role::whereNull('deleted_at')->get();
            }
            
            $departments = Department::whereNull('deleted_at')->where('status', 1)->get();
            $branch = Branch::find($employee->branch_id);
            
            return view('masters.branch.edit-branch-employee', compact('employee', 'roles', 'departments', 'branch'));
        } catch (\Exception $e) {
            Log::error('Error in BranchEmployeeController@edit: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while loading the edit form.');
        }
    }
}