<?php

namespace App\Http\Controllers;

use App\Models\Branch;
use App\Models\State;
use App\Models\District;
use App\Models\City;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class BranchController extends Controller
{
    public function index()
    {
        try {
            $branches = Branch::with(['state', 'district', 'city'])->orderBy('branch_name')->get();
            $states = State::where('status', true)->get();
            return view('masters.location.branch', compact('branches', 'states'));
        } catch (\Exception $e) {
            Log::error('Branch index error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Unable to load branches.');
        }
    }

    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'branch_name' => 'required|string|max:255',
                'branch_code' => 'required|string|max:255|unique:branches,branch_code',
                'state_id' => 'required|exists:states,id',
                'district_id' => 'required|exists:districts,id',
                'city_id' => 'required|exists:cities,id',
                'address' => 'nullable|string',
                'pincode' => 'nullable|string|max:6',
            ]);

            if ($validator->fails()) {
                Log::error('Branch validation failed: ' . json_encode($validator->errors()));
                return redirect()->back()->withErrors($validator)->withInput();
            }

            $branch = Branch::create([
                'branch_name' => $request->branch_name,
                'branch_code' => $request->branch_code,
                'state_id' => $request->state_id,
                'district_id' => $request->district_id,
                'city_id' => $request->city_id,
                'address' => $request->address,
                'pincode' => $request->pincode,
                'is_head_branch' => $request->has('is_head_branch'),
                'status' => $request->has('status') ? true : true
            ]);

            Log::info('Branch created successfully: ' . $branch->id);
            return redirect()->back()->with('success', 'Branch created successfully.');
        } catch (\Exception $e) {
            Log::error('Branch store error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Unable to create branch: ' . $e->getMessage());
        }
    }

    public function update(Request $request, Branch $branch)
    {
        try {
            $validator = Validator::make($request->all(), [
                'branch_name' => 'required|string|max:255',
                'branch_code' => 'required|string|max:255|unique:branches,branch_code,' . $branch->id,
                'state_id' => 'required|exists:states,id',
                'district_id' => 'required|exists:districts,id',
                'city_id' => 'required|exists:cities,id',
                'address' => 'nullable|string',
                'pincode' => 'nullable|string|max:6',
            ]);

            if ($validator->fails()) {
                Log::error('Branch validation failed: ' . json_encode($validator->errors()));
                return redirect()->back()->withErrors($validator)->withInput();
            }

            $branch->update([
                'branch_name' => $request->branch_name,
                'branch_code' => $request->branch_code,
                'state_id' => $request->state_id,
                'district_id' => $request->district_id,
                'city_id' => $request->city_id,
                'address' => $request->address,
                'pincode' => $request->pincode,
                'is_head_branch' => $request->has('is_head_branch'),
                'status' => $request->has('status')
            ]);

            return redirect()->back()->with('success', 'Branch updated successfully.');
        } catch (\Exception $e) {
            Log::error('Branch update error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Unable to update branch: ' . $e->getMessage());
        }
    }

    public function destroy(Branch $branch)
    {
        try {
            $branch->delete();
            return redirect()->back()->with('success', 'Branch deleted successfully.');
        } catch (\Exception $e) {
            Log::error('Branch delete error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Unable to delete branch.');
        }
    }

    public function toggleStatus(Branch $branch)
    {
        try {
            $branch->update(['status' => !$branch->status]);
            $status = $branch->status ? 'activated' : 'deactivated';
            return redirect()->back()->with('success', "Branch {$status} successfully.");
        } catch (\Exception $e) {
            Log::error('Branch toggle status error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Unable to update branch status.');
        }
    }
}