<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Applicant;   
use App\Models\User;
use App\Models\CoApplicant;
use App\Models\ApplicantGuarantor;
use App\Models\ApplicantReference;
use App\Models\ApplicantLeadStatusLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use App\Utils\FileUpload;

class RoLeadController extends Controller
{
    public function store(Request $request)
    {
        $request->validate([
            'applicant_first_name' => 'required|string|max:255',
            'applicant_last_name' => 'required|string|max:255',
            'father_name' => 'required|string|max:255',
            'contact_number' => 'required|string|max:20'
        ]);

        DB::beginTransaction();
        try {
            $applicantData = array_filter($request->only([
                'segment',
                'loan_type',
                'customer_type',
                'applicant_first_name',
                'applicant_middle_name',
                'applicant_last_name',
                'father_name',
                'caste',
                'dob',
                'age',
                'gender',
                'contact_number',
                'contact_verified',
                'email',
                'address_type',
                'current_pincode',
                'current_state',
                'current_city',
                'current_address',
                'place_of_birth',
                'nationality',
                'marital_status',
                'spouse_name',
                'pan',
                'pan_verified',
                'aadhaar',
                'aadhaar_verified',
                'form_60',
                'voter_id',
                'voter_id_verified',
                'driving_license',
                'driving_license_verified',
                'passport',
                'banking_info',
                'login_date',
                'login_amount',
                'tenure',
                'roi',
                'branch_id',
                'bm_name_id',
                'rm_name',
                'bt_yes_no_id',
                'bt_company_name',
                'imd_waiver',
                'imd_transaction_ref',
                'business_address',
                'annual_income',
                'obligation',
                'foir',
                'ltv',
                'gps_tagging',
                'property_type',
                'property_owner_name',
                'property_address',
                'psl_yes_no_id',
                'manufacturing_service_agriculture',
                'psl_sub_category',
                'purpose_of_loans',
                'psl_document_type',
                'psl_document_number',
                'field_investigation_flag',
                'tele_verification_flag',
                'personal_discussion_flag',
                'status',
                'remark',
                'assign_to'
            ]), function ($value) {
                return $value !== null && $value !== '';
            });

            // Handle applicant image upload
            if ($request->hasFile('applicant_image')) {
                $applicantData['applicant_image'] = FileUpload::uploadImage($request->file('applicant_image'), 'applicants');
            }

            $applicant = Applicant::create(array_merge($applicantData, ['created_by' => Auth::id(), 'status' => 2, 'assign_to' => Auth::id(), 'assign_date' => now(), 'assigned_by' => Auth::id()]));

            // lead create log
            ApplicantLeadStatusLog::create([
                'applicant_lead_id' => $applicant->id,
                'lead_status' => 1, //create
                'remark' => $applicant->remark,
                'action_by' => Auth::id(),
                'action_date' => now()
            ]);
            // lead assign log
            ApplicantLeadStatusLog::create([
                'applicant_lead_id' => $applicant->id,
                'lead_status' => 2, //assign
                'remark' => 'lead assigned to ' . User::find(Auth::id())->name . ' please review',
                'action_by' => Auth::id(),
                'action_date' => now()
            ]);

            // Save References
            if ($request->has('reference') && is_array($request->reference)) {
                foreach ($request->reference as $index => $referenceName) {
                    if (!empty($referenceName)) {
                        ApplicantReference::create([
                            'applicant_id' => $applicant->id,
                            'name' => $referenceName,
                            'contact_no' => $request->reference_contact_no[$index] ?? null,
                            'relation_with_applicant' => $request->reference_relation_id[$index] ?? null,
                        ]);
                    }
                }
            }

            // Save Co-Applicants
            if ($request->has('co_applicant_first_name') && is_array($request->co_applicant_first_name)) {
                foreach ($request->co_applicant_first_name as $index => $firstName) {
                    if (!empty($firstName)) {
                        $coApplicantData = array_filter([
                            'applicant_id' => $applicant->id,
                            'co_applicant_first_name' => $firstName,
                            'applicant_middle_name' => $request->co_applicant_middle_name[$index] ?? null,
                            'applicant_last_name' => $request->co_applicant_last_name[$index] ?? null,
                            'father_name' => $request->co_applicant_father_name[$index] ?? null,
                            'caste_id' => $request->co_applicant_caste[$index] ?? null,
                            'dob' => $request->co_applicant_dob[$index] ?? null,
                            'age' => $request->co_applicant_age[$index] ?? null,
                            'gender_id' => $request->co_applicant_gender[$index] ?? null,
                            'contact_number' => $request->co_applicant_contact_number[$index] ?? null,
                            'address_type_id' => $request->co_applicant_address_type[$index] ?? null,
                            'current_address' => $request->co_applicant_current_address[$index] ?? null,
                            'current_city' => $request->co_applicant_current_city[$index] ?? null,
                            'current_state' => $request->co_applicant_current_state[$index] ?? null,
                            'current_pincode' => $request->co_applicant_current_pincode[$index] ?? null,
                            'place_of_birth' => $request->co_applicant_place_of_birth[$index] ?? null,
                            'nationality' => $request->co_applicant_nationality[$index] ?? null,
                            'marital_status_id' => $request->co_applicant_marital_status[$index] ?? null,
                            'spouse_name' => $request->co_applicant_spouse_name[$index] ?? null,
                            'email' => $request->co_applicant_email[$index] ?? null,
                            'pan' => $request->co_applicant_pan[$index] ?? null,
                            'aadhaar' => $request->co_applicant_aadhaar[$index] ?? null,
                            'voter_id' => $request->co_applicant_voter_id[$index] ?? null,
                            'driving_license' => $request->co_applicant_driving_license[$index] ?? null,
                            'passport' => $request->co_applicant_passport[$index] ?? null,
                            'form_60' => $request->co_applicant_form_60[$index] ?? 0,
                        ], function ($value) {
                            return $value !== null && $value !== '';
                        });

                        // Handle co-applicant image upload
                        if ($request->hasFile('co_applicant_image') && isset($request->file('co_applicant_image')[$index])) {
                            $coApplicantData['co_applicant_image'] = FileUpload::uploadImage($request->file('co_applicant_image')[$index], 'co_applicants');
                        }

                        CoApplicant::create($coApplicantData);
                    }
                }
            }

            // Save Guarantor
            if ($request->filled('guarantor_first_name')) {
                $guarantorData = array_filter([
                    'applicant_id' => $applicant->id,
                    'guarantor_first_name' => $request->guarantor_first_name,
                    'applicant_middle_name' => $request->guarantor_middle_name,
                    'applicant_last_name' => $request->guarantor_last_name,
                    'father_name' => $request->guarantor_father_name,
                    'caste_id' => $request->guarantor_caste ?: null,
                    'dob' => $request->guarantor_dob ?: null,
                    'age' => $request->guarantor_age ?: null,
                    'gender_id' => $request->guarantor_gender ?: null,
                    'contact_number' => $request->guarantor_contact_number,
                    'address_type_id' => $request->guarantor_address_type ?: null,
                    'current_address' => $request->guarantor_current_address,
                    'current_city' => $request->guarantor_current_city,
                    'current_state' => $request->guarantor_current_state,
                    'current_pincode' => $request->guarantor_current_pincode,
                    'place_of_birth' => $request->guarantor_place_of_birth,
                    'nationality' => $request->guarantor_nationality ?: null,
                    'marital_status_id' => $request->guarantor_marital_status ?: null,
                    'spouse_name' => $request->guarantor_spouse_name,
                    'email' => $request->guarantor_email,
                    'pan' => $request->guarantor_pan,
                    'aadhaar' => $request->guarantor_aadhaar,
                    'voter_id' => $request->guarantor_voter_id,
                    'driving_license' => $request->guarantor_driving_license,
                    'passport' => $request->guarantor_passport,
                    'form_60' => $request->guarantor_form_60 ?? 0,
                ], function ($value) {
                    return $value !== null && $value !== '';
                });

                // Handle guarantor image upload
                if ($request->hasFile('applicant_guarantor_image')) {
                    $guarantorData['applicant_guarantor_image'] = FileUpload::uploadImage($request->file('applicant_guarantor_image'), 'guarantors');
                }

                ApplicantGuarantor::create($guarantorData);
            }

            DB::commit();
            return response()->json([
                'success' => true,
                'message' => 'Lead created successfully',
                'data' => ['applicant_id' => $applicant->id]
            ], 201);
        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Error in RoLeadController@store: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error creating lead'
            ], 500);
        }
    }

    public function edit(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'applicant_lead_id' => 'required|integer|exists:applicants,id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $applicant = Applicant::find($request->applicant_lead_id);

            if (!$applicant) {
                return response()->json([
                    'success' => false,
                    'message' => 'Lead not found'
                ], 404);
            }

            // Format applicant data
            $applicantData = $applicant->toArray();
            unset($applicantData['created_at'], $applicantData['updated_at'], $applicantData['deleted_at']);

            // Convert nulls to empty strings and booleans to 0/1
            foreach ($applicantData as $key => $value) {
                if ($key === 'id') continue;
                if (is_null($value)) {
                    $applicantData[$key] = '';
                } elseif (is_bool($value)) {
                    $applicantData[$key] = $value ? '1' : '0';
                } else {
                    $applicantData[$key] = (string) $value;
                }
            }

            // Get related data
            $coApplicants = $applicant->coApplicants()->get()->map(function ($coApplicant) {
                $data = $coApplicant->toArray();
                unset($data['created_at'], $data['updated_at'], $data['deleted_at']);
                foreach ($data as $key => $value) {
                    if ($key === 'id' || $key === 'applicant_id') continue;
                    if (is_null($value)) {
                        $data[$key] = '';
                    } elseif (is_bool($value)) {
                        $data[$key] = $value ? '1' : '0';
                    } else {
                        $data[$key] = (string) $value;
                    }
                }
                return $data;
            });

            $references = $applicant->references()->get()->map(function ($reference) {
                $data = $reference->toArray();
                unset($data['created_at'], $data['updated_at'], $data['deleted_at']);
                foreach ($data as $key => $value) {
                    if ($key === 'id' || $key === 'applicant_id') continue;
                    $data[$key] = is_null($value) ? '' : (string) $value;
                }
                return $data;
            });

            $guarantors = $applicant->guarantors()->get()->map(function ($guarantor) {
                $data = $guarantor->toArray();
                unset($data['created_at'], $data['updated_at'], $data['deleted_at']);
                foreach ($data as $key => $value) {
                    if ($key === 'id' || $key === 'applicant_id') continue;
                    if (is_null($value)) {
                        $data[$key] = '';
                    } elseif (is_bool($value)) {
                        $data[$key] = $value ? '1' : '0';
                    } else {
                        $data[$key] = (string) $value;
                    }
                }
                return $data;
            });

            return response()->json([
                'success' => true,
                'data' => [
                    'applicant' => $applicantData,
                    'co_applicants' => $coApplicants,
                    'references' => $references,
                    'guarantors' => $guarantors
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error in RoLeadController@edit: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error fetching lead data'
            ], 500);
        }
    }

    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'applicant_original_lead_id' => 'required|integer|exists:applicants,id',
            'applicant_first_name' => 'required|string|max:255',
            'applicant_last_name' => 'required|string|max:255',
            'father_name' => 'required|string|max:255',
            'contact_number' => 'required|string|max:20'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        try {
            $applicant = Applicant::find($request->applicant_original_lead_id);

            $applicantData = array_filter($request->only([
                'segment',
                'loan_type',
                'customer_type',
                'applicant_first_name',
                'applicant_middle_name',
                'applicant_last_name',
                'father_name',
                'caste',
                'dob',
                'age',
                'gender',
                'contact_number',
                'contact_verified',
                'email',
                'address_type',
                'current_pincode',
                'current_state',
                'current_city',
                'current_address',
                'place_of_birth',
                'nationality',
                'marital_status',
                'spouse_name',
                'pan',
                'pan_verified',
                'aadhaar',
                'aadhaar_verified',
                'form_60',
                'voter_id',
                'voter_id_verified',
                'driving_license',
                'driving_license_verified',
                'passport',
                'banking_info',
                'login_date',
                'login_amount',
                'tenure',
                'roi',
                'branch_id',
                'bm_name_id',
                'rm_name',
                'bt_yes_no_id',
                'bt_company_name',
                'imd_waiver',
                'imd_transaction_ref',
                'business_address',
                'annual_income',
                'obligation',
                'foir',
                'ltv',
                'gps_tagging',
                'property_type',
                'property_owner_name',
                'property_address',
                'psl_yes_no_id',
                'manufacturing_service_agriculture',
                'psl_sub_category',
                'purpose_of_loans',
                'psl_document_type',
                'psl_document_number',
                'field_investigation_flag',
                'tele_verification_flag',
                'personal_discussion_flag',
                'status',
                'remark',
                'assign_to'
            ]), function ($value) {
                return $value !== null && $value !== '';
            });

            // Handle applicant image upload
            if ($request->hasFile('applicant_image')) {
                $applicantData['applicant_image'] = FileUpload::uploadImage($request->file('applicant_image'), 'applicants');
            }

            if ($request->status == 3) {
                // lead assign log
                ApplicantLeadStatusLog::create([
                    'applicant_lead_id' => $applicant->id,
                    'lead_status' => 3, //ro verified
                    'action_by' => Auth::id(),
                    'action_date' => now(),
                    'remark' => $request->remark
                ]);
                $applicantData['status'] = 3; // RO Verified
            }

            $applicant->update($applicantData);

            // Update References - Update existing instead of delete/recreate
            $existingReferences = $applicant->references()->get()->keyBy('id');
            $processedReferenceIds = [];

            if ($request->has('reference') && is_array($request->reference)) {
                foreach ($request->reference as $index => $referenceName) {
                    if (!empty($referenceName)) {
                        $referenceData = [
                            'name' => $referenceName,
                            'contact_no' => $request->reference_contact_no[$index] ?? null,
                            'relation_with_applicant' => $request->reference_relation_id[$index] ?? null,
                        ];

                        // Check if this is an update (has reference_id) or new record
                        if (isset($request->reference_id[$index]) && $existingReferences->has($request->reference_id[$index])) {
                            $existingReferences->get($request->reference_id[$index])->update($referenceData);
                            $processedReferenceIds[] = $request->reference_id[$index];
                        } else {
                            $newReference = ApplicantReference::create(array_merge($referenceData, ['applicant_id' => $applicant->id]));
                            $processedReferenceIds[] = $newReference->id;
                        }
                    }
                }
            }

            // Delete references that are no longer in the request
            $applicant->references()->whereNotIn('id', $processedReferenceIds)->delete();

            // Update Co-Applicants - Update existing instead of delete/recreate
            $existingCoApplicants = $applicant->coApplicants()->get()->keyBy('id');
            $processedCoApplicantIds = [];

            if ($request->has('co_applicant_first_name') && is_array($request->co_applicant_first_name)) {
                foreach ($request->co_applicant_first_name as $index => $firstName) {
                    if (!empty($firstName)) {
                        $coApplicantData = array_filter([
                            'co_applicant_first_name' => $firstName,
                            'applicant_middle_name' => $request->co_applicant_middle_name[$index] ?? null,
                            'applicant_last_name' => $request->co_applicant_last_name[$index] ?? null,
                            'father_name' => $request->co_applicant_father_name[$index] ?? null,
                            'caste_id' => $request->co_applicant_caste[$index] ?? null,
                            'dob' => $request->co_applicant_dob[$index] ?? null,
                            'age' => $request->co_applicant_age[$index] ?? null,
                            'gender_id' => $request->co_applicant_gender[$index] ?? null,
                            'contact_number' => $request->co_applicant_contact_number[$index] ?? null,
                            'address_type_id' => $request->co_applicant_address_type[$index] ?? null,
                            'current_address' => $request->co_applicant_current_address[$index] ?? null,
                            'current_city' => $request->co_applicant_current_city[$index] ?? null,
                            'current_state' => $request->co_applicant_current_state[$index] ?? null,
                            'current_pincode' => $request->co_applicant_current_pincode[$index] ?? null,
                            'place_of_birth' => $request->co_applicant_place_of_birth[$index] ?? null,
                            'nationality' => $request->co_applicant_nationality[$index] ?? null,
                            'marital_status_id' => $request->co_applicant_marital_status[$index] ?? null,
                            'spouse_name' => $request->co_applicant_spouse_name[$index] ?? null,
                            'email' => $request->co_applicant_email[$index] ?? null,
                            'pan' => $request->co_applicant_pan[$index] ?? null,
                            'aadhaar' => $request->co_applicant_aadhaar[$index] ?? null,
                            'voter_id' => $request->co_applicant_voter_id[$index] ?? null,
                            'driving_license' => $request->co_applicant_driving_license[$index] ?? null,
                            'passport' => $request->co_applicant_passport[$index] ?? null,
                            'form_60' => $request->co_applicant_form_60[$index] ?? 0,
                        ], function ($value) {
                            return $value !== null && $value !== '';
                        });

                        // Handle co-applicant image upload
                        if ($request->hasFile('co_applicant_image') && isset($request->file('co_applicant_image')[$index])) {
                            $coApplicantData['co_applicant_image'] = FileUpload::uploadImage($request->file('co_applicant_image')[$index], 'co_applicants');
                        }

                        // Check if this is an update (has co_applicant_id) or new record
                        if (isset($request->co_applicant_id[$index]) && $existingCoApplicants->has($request->co_applicant_id[$index])) {
                            $existingCoApplicants->get($request->co_applicant_id[$index])->update($coApplicantData);
                            $processedCoApplicantIds[] = $request->co_applicant_id[$index];
                        } else {
                            $newCoApplicant = CoApplicant::create(array_merge($coApplicantData, ['applicant_id' => $applicant->id]));
                            $processedCoApplicantIds[] = $newCoApplicant->id;
                        }
                    }
                }
            }

            // Delete co-applicants that are no longer in the request
            $applicant->coApplicants()->whereNotIn('id', $processedCoApplicantIds)->delete();

            // Update Guarantor - Update existing instead of delete/recreate
            $existingGuarantor = $applicant->guarantors()->first();

            if ($request->filled('guarantor_first_name')) {
                $guarantorData = array_filter([
                    'guarantor_first_name' => $request->guarantor_first_name,
                    'applicant_middle_name' => $request->guarantor_middle_name,
                    'applicant_last_name' => $request->guarantor_last_name,
                    'father_name' => $request->guarantor_father_name,
                    'caste_id' => $request->guarantor_caste ?: null,
                    'dob' => $request->guarantor_dob ?: null,
                    'age' => $request->guarantor_age ?: null,
                    'gender_id' => $request->guarantor_gender ?: null,
                    'contact_number' => $request->guarantor_contact_number,
                    'address_type_id' => $request->guarantor_address_type ?: null,
                    'current_address' => $request->guarantor_current_address,
                    'current_city' => $request->guarantor_current_city,
                    'current_state' => $request->guarantor_current_state,
                    'current_pincode' => $request->guarantor_current_pincode,
                    'place_of_birth' => $request->guarantor_place_of_birth,
                    'nationality' => $request->guarantor_nationality ?: null,
                    'marital_status_id' => $request->guarantor_marital_status ?: null,
                    'spouse_name' => $request->guarantor_spouse_name,
                    'email' => $request->guarantor_email,
                    'pan' => $request->guarantor_pan,
                    'aadhaar' => $request->guarantor_aadhaar,
                    'voter_id' => $request->guarantor_voter_id,
                    'driving_license' => $request->guarantor_driving_license,
                    'passport' => $request->guarantor_passport,
                    'form_60' => $request->guarantor_form_60 ?? 0,
                ], function ($value) {
                    return $value !== null && $value !== '';
                });

                // Handle guarantor image upload
                if ($request->hasFile('guarantor_image')) {
                    $guarantorData['applicant_guarantor_image'] = FileUpload::uploadImage($request->file('guarantor_image'), 'guarantors');
                }

                if ($existingGuarantor) {
                    $existingGuarantor->update($guarantorData);
                } else {
                    ApplicantGuarantor::create(array_merge($guarantorData, ['applicant_id' => $applicant->id]));
                }
            } else {
                // If no guarantor data provided, delete existing guarantor
                if ($existingGuarantor) {
                    $existingGuarantor->delete();
                }
            }

            DB::commit();
            return response()->json([
                'success' => true,
                'message' => 'Lead updated successfully',
                'data' => ['applicant_id' => $applicant->id]
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Error in RoLeadController@update: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error updating lead'
            ], 500);
        }
    }
}
