<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Branch;
use App\Models\Role;
use App\Models\Department;
use App\Models\FormField;
use App\Services\ApiLogService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class MasterController extends Controller
{
    public function masterUsers(Request $request)
    {
        // Entry log
        ApiLogService::entry('Master users request received', $request->all());

        try {
            // Get roles with branches and users (exclude admin role)
            $roles = Role::where('id', '!=', 1)
                ->with([
                    'users' => function ($query) use ($request) {
                        $query->whereNull('deleted_at')
                            ->where('status', 1);

                        if ($request->filled('branch_id')) {
                            $query->where('branch_id', $request->branch_id);
                        }

                        if ($request->filled('department_id')) {
                            $query->where('department_id', $request->department_id);
                        }
                    },
                    'users.branch:id,branch_name,branch_code',
                    'users.department:id,name'
                ])->get();

            // Filter roles that have users
            if ($request->filled('role_id')) {
                $roles = $roles->where('id', $request->role_id);
            }

            // Group users by role, branch, and department
            $rolesData = [];
            foreach ($roles as $role) {
                if ($role->users->isEmpty()) continue;

                $branchesData = [];
                $usersByBranch = $role->users->groupBy('branch_id');

                foreach ($usersByBranch as $branchId => $users) {
                    $branch = $users->first()->branch;

                    $departmentsData = [];
                    $usersByDepartment = $users->groupBy('department_id');

                    foreach ($usersByDepartment as $departmentId => $departmentUsers) {
                        $department = $departmentUsers->first()->department;

                        $usersData = [];
                        foreach ($departmentUsers as $user) {
                            $usersData[] = [
                                'id' => $user->id,
                                'staff_id' => (string)($user->staff_id ?? ''),
                                'role_id' => (string)($user->role_id ?? ''),
                                'name' => (string)($user->name ?? ''),
                                'father_name' => (string)($user->father_name ?? ''),
                                'mother_name' => (string)($user->mother_name ?? ''),
                                'dob' => (string)($user->dob ?? ''),
                                'gender' => (string)($user->gender ?? ''),
                                'email' => (string)($user->email ?? ''),
                                'mobile' => (string)($user->mobile ?? ''),
                                'aadhaar_number' => (string)($user->aadhaar_number ?? ''),
                                'pan_number' => (string)($user->pan_number ?? ''),
                                'joining_date' => (string)($user->joining_date ?? ''),
                                'profile_pic' => $user->profile_pic ? asset('storage/uploads/profiles/' . $user->profile_pic) : '',
                                'addhar_front_pic' => $user->addhar_front_pic ? asset('storage/uploads/documents/' . $user->addhar_front_pic) : '',
                                'addhar_back_pic' => $user->addhar_back_pic ? asset('storage/uploads/documents/' . $user->addhar_back_pic) : '',
                                'status' => (string)($user->status ?? '')
                            ];
                        }

                        $departmentsData[] = [
                            'id' => $department->id ?? "",
                            'name' => (string)($department->name ?? ''),
                            'users' => $usersData
                        ];
                    }

                    $branchesData[] = [
                        'id' => $branch->id ?? "",
                        'branch_name' => (string)($branch->branch_name ?? ''),
                        'branch_code' => (string)($branch->branch_code ?? ''),
                        'departments' => $departmentsData
                    ];
                }

                $rolesData[] = [
                    'id' => $role->id,
                    'name' => (string)($role->name ?? ''),
                    'branches' => $branchesData
                ];
            }

            $response = [
                'status' => true,
                'data' => [
                    'roles' => $rolesData
                ]
            ];

            // Success log
            ApiLogService::success('Master users retrieved successfully', [
                'total_roles' => count($rolesData)
            ]);

            // Exit log
            ApiLogService::exit('Master users request completed successfully', $response);

            return response()->json($response);
        } catch (\Exception $e) {
            // Error log
            Log::error('Error in MasterController@masterUsers: ' . $e->getMessage(), [
                'request' => $request->all(),
                'trace' => $e->getTraceAsString()
            ]);

            ApiLogService::error('Master users request failed with exception', $e);

            return response()->json([
                'status' => false,
                'message' => 'An error occurred while fetching users.'
            ], 500);
        }
    }

    public function formMasterFields(Request $request)
    {
        // Entry log
        ApiLogService::entry('Form master fields request received', $request->all());

        try {
            $formFields = FormField::with('options')->get();
            
            $formattedFields = [];
            foreach ($formFields as $field) {
                $formattedOptions = [];
                foreach ($field->options as $option) {
                    $formattedOptions[] = [
                        'id' => $option->id,
                        'field_id' => (string)($option->field_id ?? ''),
                        'option_value' => (string)($option->option_value ?? ''),
                        'priority' => (string)($option->priority ?? '')
                    ];
                }
                
                $formattedFields[$field->field_name] = [
                    'id' => $field->id,
                    'field_name' => (string)($field->field_name ?? ''),
                    'field_label' => (string)($field->field_label ?? ''),
                    'field_type' => (string)($field->field_type ?? ''),
                    'options' => $formattedOptions
                ];
            }

            $response = [
                'status' => true,
                'data' => $formattedFields
            ];

            // Success log
            ApiLogService::success('Form master fields retrieved successfully', [
                'total_fields' => count($formattedFields)
            ]);

            // Exit log
            ApiLogService::exit('Form master fields request completed successfully', $response);

            return response()->json($response);
        } catch (\Exception $e) {
            // Error log
            Log::error('Error in MasterController@formMasterFields: ' . $e->getMessage(), [
                'request' => $request->all(),
                'trace' => $e->getTraceAsString()
            ]);

            ApiLogService::error('Form master fields request failed with exception', $e);

            return response()->json([
                'status' => false,
                'message' => 'An error occurred while fetching form fields.'
            ], 500);
        }
    }
}
