<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Services\ApiLogService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class AuthController extends Controller
{
    public function login(Request $request)
    {
        // Entry log
        ApiLogService::entry('Login request received', $request->only(['email', 'deviceInfo']));

        try {
            // Rate limiting
            $key = 'login.' . $request->ip();
            if (RateLimiter::tooManyAttempts($key, 5)) {
                ApiLogService::error('Too many login attempts', ['email' => $request->email]);

                return response()->json([
                    'status' => false,
                    'message' => 'Too many login attempts. Please try again later.'
                ], 429);
            }

            $validator = Validator::make($request->all(), [
                'email' => 'required|string|max:255',
                'password' => 'required|string|min:6|max:255',
                'deviceInfo' => 'nullable|string|max:500',
            ]);

            if ($validator->fails()) {
                ApiLogService::error('Validation failed', $validator->errors()->toArray());

                return response()->json([
                    'status' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Optimized query with role relationship - support both email and mobile login
            $user = User::with('role')
                ->where(function ($query) use ($request) {
                    $query->where('email', $request->email)
                        ->orWhere('mobile', $request->email);
                })
                ->where('role_id', 4) // Only Recovery Officers
                ->where('status', 1) // Only active users (1 = active)
                ->first();

            if (!$user || !Hash::check($request->password, $user->password)) {
                RateLimiter::hit($key, 300); // 5 minutes lockout

                ApiLogService::error('Invalid credentials provided', ['email' => $request->email]);

                Log::warning('Failed login attempt', [
                    'email' => $request->email,
                    'ip' => $request->ip(),
                    'user_agent' => $request->userAgent()
                ]);

                return response()->json([
                    'status' => false,
                    'message' => 'Invalid credentials or account not authorized.'
                ], 401);
            }

            // Check if user is already logged in (has active tokens)
            $existingTokens = $user->tokens()->where('name', 'recovery-officer-token')->count();
            if ($existingTokens > 0) {
                ApiLogService::warning('User already logged in', [
                    'user_id' => $user->id,
                    'existing_tokens' => $existingTokens
                ]);

                // Optionally revoke existing tokens for single session
                $user->tokens()->where('name', 'recovery-officer-token')->delete();
            }

            // Clear rate limit on successful login
            RateLimiter::clear($key);

            // Create token with expiration
            $token = $user->createToken('recovery-officer-token', ['*'], now()->addHours(8))->plainTextToken;

            // Update device info and last login
            $user->update([
                'device_info' => $request->deviceInfo ?: null,
                'last_login_at' => now(),
                'last_login_ip' => $request->ip() ?: 'unknown'
            ]);

            $response = [
                'status' => true,
                'message' => "Hi {$user->name}! You have been logged in as {$user->role->name}.",
                'token' => $token,
                'user_id' => (string)$user->id,
                'role' => $user->role->name,
                'name' => $user->name
            ];

            // Success log
            ApiLogService::success('User successfully logged in', [
                'user_id' => $user->id,
                'email' => $user->email,
                'role' => $user->role->name
            ]);

            Log::info('Successful login', [
                'user_id' => $user->id,
                'email' => $user->email,
                'ip' => $request->ip()
            ]);

            // Exit log
            ApiLogService::exit('Login request completed successfully', $response);

            return response()->json($response);
        } catch (\Exception $e) {
            // Error log for exceptions
            ApiLogService::error('Login process failed with exception', $e);

            Log::error('Login exception', [
                'error' => $e->getMessage(),
                'email' => $request->email ?? 'unknown',
                'ip' => $request->ip()
            ]);

            return response()->json([
                'status' => false,
                'message' => 'An error occurred during login. Please try again.'
            ], 500);
        }
    }

    public function logout(Request $request)
    {
        // Entry log
        ApiLogService::entry('Logout request received');

        try {
            $user = $request->user();
            $token = $request->user()->currentAccessToken();

            $token->delete();

            // Success log
            ApiLogService::success('User successfully logged out', [
                'user_id' => $user->id,
                'email' => $user->email
            ]);

            // Exit log
            ApiLogService::exit('Logout request completed successfully');

            return response()->json([
                'status' => true,
                'message' => 'Logged out successfully'
            ]);
        } catch (\Exception $e) {
            // Error log for exceptions
            ApiLogService::error('Logout process failed with exception', $e);

            return response()->json([
                'status' => false,
                'message' => 'An error occurred during logout.'
            ], 500);
        }
    }

    public function user(Request $request)
    {
        // Entry log
        ApiLogService::entry('User info request received');

        try {
            $user = $request->user();

            $response = [
                'user' => $user
            ];

            // Success log
            ApiLogService::success('User info retrieved successfully', [
                'user_id' => $user->id,
                'email' => $user->email
            ]);

            // Exit log
            ApiLogService::exit('User info request completed successfully', $response);

            return response()->json($response);
        } catch (\Exception $e) {
            // Error log for exceptions
            ApiLogService::error('User info request failed with exception', $e);

            return response()->json([
                'status' => false,
                'message' => 'An error occurred while retrieving user information.'
            ], 500);
        }
    }
}
